/*
* Copyright 2008 Jay Clegg.  All rights reserved.
*
*    This program is free software: you can redistribute it and/or modify
*    it under the terms of the GNU General Public License as published by
*    the Free Software Foundation, either version 3 of the License, or
*    (at your option) any later version.
*
*    This program is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*    GNU General Public License for more details.
*
*    You should have received a copy of the GNU General Public License
*    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

//
//  QCSerialPatchPlugIn.m
//  QCSerialPatch
//
//  Created by jay on 6/22/08.
//  Copyright 2008 Jay Clegg. All rights reserved.
//

/* It's highly recommended to use CGL macros instead of changing the current context for plug-ins that perform OpenGL rendering */
#import <OpenGL/CGLMacro.h>

#include <fcntl.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <paths.h>
#include <termios.h>
#include <sysexits.h>

#import "Serial.h"
#import "QCSerialPatchPlugIn.h"

int _serialPortDescriptor = -1;

#define BAUD_RATE 115200


#define	kQCPlugIn_Name				@"Serial to Peggy2"
#define kQCPlugIn_Ver				@"1.0"
#define	kQCPlugIn_Description		kQCPlugIn_Name @" v" kQCPlugIn_Ver @" patch by Jay Clegg. Outputs to a 'Peggy 2.0' that has been modified to accept serial input."
#define DefaultSerialPort			@"/dev/tty.usbserial-XXXXXXXX"

@implementation QCSerialPatchPlugIn

// declare the input / output properties as dynamic as Quartz Composer will handle their implementation
@dynamic inputSerialPort, inputImage;
//@dynamic inputVersion;

+ (NSDictionary*) attributes
{
	//Return a dictionary of attributes describing the plug-in (QCPlugInAttributeNameKey, QCPlugInAttributeDescriptionKey...).
	return [NSDictionary dictionaryWithObjectsAndKeys:kQCPlugIn_Name, QCPlugInAttributeNameKey, kQCPlugIn_Description, QCPlugInAttributeDescriptionKey, nil];
}


+ (NSDictionary*) attributesForPropertyPortWithKey:(NSString*)key
{
	//Specify the optional attributes for property based ports (QCPortAttributeNameKey, QCPortAttributeDefaultValueKey...).
	
	if([key isEqualToString:@"inputImage"])
		return [NSDictionary dictionaryWithObjectsAndKeys:@"Image", QCPortAttributeNameKey, nil];
	if([key isEqualToString:@"inputSerialPort"])
		return [NSDictionary dictionaryWithObjectsAndKeys:@"Serial Port", QCPortAttributeNameKey, DefaultSerialPort, QCPortAttributeDefaultValueKey, nil];
	if ([key isEqualToString: @"inputVersion"])
		return [NSDictionary dictionaryWithObjectsAndKeys:@"Version", QCPortAttributeNameKey, kQCPlugIn_Ver,  QCPortAttributeDefaultValueKey, nil];
		
//	if([key isEqualToString:@"outputConnected"])
//		return [NSDictionary dictionaryWithObjectsAndKeys:@"Serial Connected", QCPortAttributeNameKey, NO, QCPortAttributeDefaultValueKey, nil];
	
	
	return nil;
}

+ (QCPlugInExecutionMode) executionMode
{
	//Return the execution mode of the plug-in: kQCPlugInExecutionModeProvider, kQCPlugInExecutionModeProcessor, or kQCPlugInExecutionModeConsumer.
	return kQCPlugInExecutionModeConsumer;
}

+ (QCPlugInTimeMode) timeMode
{
	//Return the time dependency mode of the plug-in: kQCPlugInTimeModeNone, kQCPlugInTimeModeIdle or kQCPlugInTimeModeTimeBase.
	return kQCPlugInTimeModeNone;
}

- (id) init
{
	if(self = [super init]) 
	{
		// allocate any non-transient resouces required
	}
	
	return self;
}

- (void) finalize
{
	//Release any non garbage collected resources created in -init.
	[super finalize];
}

- (void) dealloc
{
	// release resources created by -init
	[super dealloc];
}


@end





BOOL SendFrame(CGImageRef cgImage)
{
	unsigned char header[6] = {0xde, 0xad, 0xbe, 0xef,1,0 };
	int i,j,k;
	unsigned char buffer[13];

	//int width = CGImageGetWidth(cgImage);
	//int height = CGImageGetHeight(cgImage);
	
	// get a 25x25 bitmap context
	int width = 25;
	int height = 25;
	
	unsigned char *data = malloc(width * height * sizeof(unsigned char) * 4);
	CGContextRef bitmapContext = CGBitmapContextCreate(data,
							   width,
							   height,
							   8,
							   width * 4,
							   CGColorSpaceCreateDeviceRGB(),
							   kCGImageAlphaPremultipliedLast);
	if (bitmapContext)
	{						   

		// copy the current image into the 25x25 bitmap context
		CGContextDrawImage(bitmapContext, CGRectMake(0,0,width,height), cgImage);
		unsigned char *bitmap = CGBitmapContextGetData(bitmapContext);
		if (bitmap)
		{
		
			sendSerialBytes(_serialPortDescriptor, header, sizeof(header));

			int pair;
			for ( i =0; i < 25; i++)
			{
				for ( j =0; j < 13; j++)
				{
					pair = 0;
					for ( k =0; k < 2; k++)
					{
						int pixelNum = i * 25 + j * 2 + (1-k);
						
						int blu = (int)(unsigned int)bitmap[pixelNum * 4 + 0];
						int grn = (int)(unsigned int)bitmap[pixelNum * 4 + 1];
						int red = (int)(unsigned int)bitmap[pixelNum * 4 + 2];
						int alp = (int)(unsigned int)bitmap[pixelNum * 4 + 3];
						//int gray = ((red + grn + blu) /3) >>4;
						
						red = (alp * red)>>8;
						grn = (alp * grn)>>8;
						blu = (alp * blu)>>8;
						
						int gray = ((red + grn + blu)/3) >> 4;

						if (gray > 15) gray = 15;
						else if (gray < 0) gray = 0;
					
						pair = (pair <<4 | gray);
					}
					
					buffer[j]= (unsigned char) (pair & 0xff);
				}
				sendSerialBytes(_serialPortDescriptor, buffer, sizeof(buffer));
			}
			flushSerial(_serialPortDescriptor);
		}

		CGContextRelease(bitmapContext);
		free(data);

	}
	return YES;

}

@implementation QCSerialPatchPlugIn (Execution)


- (BOOL) startExecution:(id<QCPlugInContext>)context
{
	/*
	Called by Quartz Composer when rendering of the composition starts: perform any required setup for the plug-in.
	Return NO in case of fatal failure (this will prevent rendering of the composition to start).
	*/
	//self.outputConnected = (_serialPortDescriptor == -1 ? NO : YES);
	return YES;
}

- (void) enableExecution:(id<QCPlugInContext>)context
{
	/*
	Called by Quartz Composer when the plug-in instance starts being used by Quartz Composer.
	*/
	//self.outputConnected = (_serialPortDescriptor == -1 ? NO : YES);
}

- (BOOL) execute:(id<QCPlugInContext>)context atTime:(NSTimeInterval)time withArguments:(NSDictionary*)arguments
{
	/*
	Called by Quartz Composer whenever the plug-in instance needs to execute.
	Only read from the plug-in inputs and produce a result (by writing to the plug-in outputs or rendering to the destination OpenGL context) within that method and nowhere else.
	Return NO in case of failure during the execution (this will prevent rendering of the current frame to complete).
	
	The OpenGL context for rendering can be accessed and defined for CGL macros using:
	CGLContextObj cgl_ctx = [context CGLContextObj];
	*/
	
	id<QCPlugInInputImageSource>	qcImage = self.inputImage;
	NSString*						pixelFormat;
	CGColorSpaceRef					colorSpace;
	CGDataProviderRef				dataProvider;
	CGImageRef						cgImage;


	//self.outputVersion = kQCPlugIn_Ver;
	
	/* Make sure we have a new image */
	if(![self didValueForInputKeyChange:@"inputImage"] || !qcImage || ![self.inputSerialPort length])
    	return YES;
		
	
	if (![self.inputSerialPort length]) return NO;
	NSString * port = self.inputSerialPort;
	if (_serialPortDescriptor == -1)
	{
		//self.outputConnected = NO;
		_serialPortDescriptor = openSerialPort( [ port UTF8String], BAUD_RATE);
		if (-1 == _serialPortDescriptor)
		{
			//NSLog(@"Couldn't open serial port");
			return NO;
		}
		else
		{
			//NSLog(@"Connected to serial port");
		}
	}
	
	
	if (_serialPortDescriptor == -1) return NO;
	
	
	/* Figure out pixel format and colorspace to use */
	colorSpace = [qcImage imageColorSpace];
	if(CGColorSpaceGetModel(colorSpace) == kCGColorSpaceModelMonochrome)
		pixelFormat = QCPlugInPixelFormatI8;
	else if(CGColorSpaceGetModel(colorSpace) == kCGColorSpaceModelRGB)
		pixelFormat = QCPlugInPixelFormatARGB8;
	else
		return NO;
	
	//NSLog(@"Pixel format = %d", pixelFormat == QCPlugInPixelFormatARGB8 ? 32 : 8);
		
	/* Get a buffer representation from the image in its native colorspace */
	if(![qcImage lockBufferRepresentationWithPixelFormat:pixelFormat colorSpace:colorSpace forBounds:[qcImage imageBounds]])
	return NO;
	
	/* Create CGImage from buffer */
	dataProvider = CGDataProviderCreateWithData(NULL, [qcImage bufferBaseAddress], [qcImage bufferPixelsHigh] * [qcImage bufferBytesPerRow], NULL);
	cgImage = CGImageCreate([qcImage bufferPixelsWide], [qcImage bufferPixelsHigh], 8, (pixelFormat == QCPlugInPixelFormatI8 ? 8 : 32),[qcImage bufferBytesPerRow], 
				colorSpace, (pixelFormat == QCPlugInPixelFormatI8 ? 0 : /*kCGImageAlphaPremultipliedFirst |*/ kCGBitmapByteOrder32Host), 
				dataProvider, NULL, false, kCGRenderingIntentDefault);
	CGDataProviderRelease(dataProvider);
	if(cgImage == NULL) {
		[qcImage unlockBufferRepresentation];
		return NO;
	}
	
	SendFrame(cgImage);

	/* Destroy CGImage */
	CGImageRelease(cgImage);
	
	/* Release buffer representation */
	[qcImage unlockBufferRepresentation];
	
	return YES;
}


/*
Called by Quartz Composer when the plug-in instance stops being used by Quartz Composer.
*/

- (void) disableExecution:(id<QCPlugInContext>)context
{
	if (_serialPortDescriptor != -1)
		closeSerialPort(_serialPortDescriptor);
	_serialPortDescriptor = -1;
}

/*
Called by Quartz Composer when rendering of the composition stops: perform any required cleanup for the plug-in.
*/
- (void) stopExecution:(id<QCPlugInContext>)context
{
	if (_serialPortDescriptor != -1)
		closeSerialPort(_serialPortDescriptor);
	_serialPortDescriptor = -1;
}

@end
