/*
* Copyright 2008 Jay Clegg.  All rights reserved.
*
*    This program is free software: you can redistribute it and/or modify
*    it under the terms of the GNU General Public License as published by
*    the Free Software Foundation, either version 3 of the License, or
*    (at your option) any later version.
*
*    This program is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*    GNU General Public License for more details.
*
*    You should have received a copy of the GNU General Public License
*    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


#include <IOKit/IOKitLib.h>
#include <IOKit/serial/IOSerialKeys.h>
#include <IOKit/serial/ioss.h>
#include <IOKit/IOBSD.h>
#include "Serial.h"

// Hold the original termios attributes so we can reset them
static struct termios gOriginalTTYAttrs;


// open a serial device, return the file descriptor id for it.
// code for this method derived from the SerialPortSample.c source code
// from Apple


int openSerialPort(const char *path, int baudRate)
{
    int	serialPortDescriptor = -1;
    
    // Open the serial port read/write, with no controlling terminal, and don't wait for a connection.
    // The O_NONBLOCK flag also causes subsequent I/O on the device to be non-blocking.
    // See open(2) ("man 2 open") for details.
    serialPortDescriptor = open(path, O_RDWR | O_NOCTTY | O_NONBLOCK);
    if (serialPortDescriptor == -1)
    {
        NSLog(@"Error opening serial port %s - %s(%d).\n", path, strerror(errno), errno);
        goto error;
    }

    // Note that open() follows POSIX semantics: multiple open() calls to the same file will succeed
    // unless the TIOCEXCL ioctl is issued. This will prevent additional opens except by root-owned
    // processes.
    
    if (ioctl(serialPortDescriptor, TIOCEXCL) == -1)
    {
        NSLog(@"Error setting TIOCEXCL on %s - %s(%d).\n",path, strerror(errno), errno);
        goto error;
    }
    
    // Now that the device is open, clear the O_NONBLOCK flag so subsequent I/O will block.
    if (fcntl(serialPortDescriptor, F_SETFL, 0) == -1)
    {
        NSLog(@"Error clearing O_NONBLOCK %s - %s(%d).\n", path, strerror(errno), errno);
        goto error;
    }
    
    // Get the current options and save them so we can restore the default settings later.
    if (tcgetattr(serialPortDescriptor, &gOriginalTTYAttrs) == -1)
    {
        NSLog(@"Error getting tty attributes %s - %s(%d).\n", path, strerror(errno), errno);
        goto error;
    }

    // The serial port attributes such as timeouts and baud rate are set by modifying the termios
    // structure and then calling tcsetattr() to cause the changes to take effect. Note that the
    // changes will not become effective without the tcsetattr() call.
    // See tcsetattr(4) ("man 4 tcsetattr") for details.

    struct termios	options;
    options = gOriginalTTYAttrs;
    
    // Print the current input and output baud rates.
    //printf("Current input baud rate is %d\n", (int) cfgetispeed(&options));
    //printf("Current output baud rate is %d\n", (int) cfgetospeed(&options));
    
    // Set raw input (non-canonical) mode, with reads blocking until either a single character 
    // has been received or a one second timeout expires.
    // See tcsetattr(4) ("man 4 tcsetattr") and termios(4) ("man 4 termios") for details.
    
    cfmakeraw(&options);
    options.c_cc[VMIN] = 1;
    options.c_cc[VTIME] = 10;
        
    // The baud rate, word length, and handshake options can be set as follows:
    //cfsetspeed(&options, B38400);		// Set 19200 baud    
	
    options.c_cflag |= CS8; 	  // Use 8 bit words
	options.c_cflag |= CLOCAL;       // Ignore status lines
	options.c_cflag |= CREAD;
	//options.c_cflag &= ~CRTSCTS; // Disable RTS/CTS
	//options.c_cflag |= PARENB	    	// Parity enable (even parity if PARODD not also set)
	//options.c_cflag |= CCTS_OFLOW;  	// CTS flow control of output
	//options.c_cflag |= CRTS_IFLOW	    // RTS flow control of input
				
   options.c_oflag &= ~OPOST; // No output processing
   //options.c_oflag &= ~ONLCR; // Don't convert linefeeds
   //options.c_lflag &= ~ISIG;

   options.c_lflag &= ~(ICANON | ECHO | ECHOE | ISIG);
   
    // Cause the new options to take effect immediately.
    if (tcsetattr(serialPortDescriptor, TCSANOW, &options) == -1)
    {
        NSLog(@"Error setting tty attributes %s - %s(%d).\n", path, strerror(errno), errno);
        goto error;
    }
	
	
			
	// set input/output baudrate
	speed_t speed = baudRate; 
    if (ioctl(serialPortDescriptor, IOSSIOSPEED, &speed) == -1)
    {
        NSLog(@"Error calling ioctl(, IOSSIOSPEED,) %s - %s(%d).\n", path, strerror(errno), errno);
    }
	
	
	NSLog(@"Serial Port configured successfully\n");

    // Success
    return serialPortDescriptor;
    
	// Fail
error:
    if (serialPortDescriptor != -1)
    {
        close(serialPortDescriptor);
    }
    
    return -1;
}




void sendSerialBytes(int serialPortDescriptor, unsigned char * data, int len)
{
	int numBytes;
	numBytes = write(serialPortDescriptor, data, len);
	if (numBytes == -1)
	{
		NSLog(@"Error writing to serial port - %s(%d).\n", strerror(errno), errno);
	}
	else if (numBytes != len)
	{
		NSLog(@"Written bytes mismatch, Wrote %ld bytes, but expected %ld\n", numBytes, len);
	}
							
}

void flushSerial(int serialPortDescriptor)
{
	//ioctl(serialPortDescriptor, TIOCFLUSH, (FREAD | FWRITE ));
    if (tcdrain(serialPortDescriptor) == -1)
    {
        //NSLog(@"Error waiting for drain - %s(%d).\n", strerror(errno), errno);
    }
}


// Given the file descriptor for a serial device, close that device.
void closeSerialPort(int serialPortDescriptor)
{
    // Block until all written output has been sent from the device.
    // Note that this call is simply passed on to the serial device driver. 
	// See tcsendbreak(3) ("man 3 tcsendbreak") for details.
    if (tcdrain(serialPortDescriptor) == -1)
    {
        NSLog(@"Error waiting for drain - %s(%d).\n", strerror(errno), errno);
    }
    
    // Traditionally it is good practice to reset a serial port back to
    // the state in which you found it. This is why the original termios struct
    // was saved.
    if (tcsetattr(serialPortDescriptor, TCSANOW, &gOriginalTTYAttrs) == -1)
    {
        NSLog(@"Error resetting tty attributes - %s(%d).\n",strerror(errno), errno);
    }

    close(serialPortDescriptor);
	NSLog(@"serial port closed");
}



